/*
Author Dumitru Novitchi
 */

 /*

    LOW FUSE      0xE2
    HIGH FUSE     0xDC
    LOCK BITS     0xFF
    EXTENDED FUSE 0xFF

 */

#define F_CPU 8000000UL // 8Mhz internal

#include <avr/io.h>
#include <avr/interrupt.h>
#include <stdlib.h>
#include <avr/eeprom.h>
#include "HD44780.h"
#include "int_to_char.h"

#define Voltage_Chanel ADMUX = ( 1 << MUX1 ) | ( 1 << MUX2 ) // Diferential mode between PB4 -> PB3
#define Current_Chanel ADMUX = ( 1 << MUX0 ) // Current PB2
#define ADC_Start ADCSRA |= ( 1 << ADSC )
#define ADC_OFF ADCSRA &= ~( 1 << ADIE )

#define min_voltage 0
#define max_voltage 299
#define min_current 0
#define max_current 45
#define Gain 90
#define Divide 6

// calibration -----------------------
#define Ref_Volt  5250
//-------------------------------------
//#define LCD_I2C_addr 0x4E         // PCF8574T = 0x4E , PCF8574AT = 0x7E set in library HD44780.h
#define Key_I2C_Adress 0x40
#define MCP4725_Voltage_I2C_Adress 0xC2
#define MCP4725_Current_I2C_Adress 0xC0
#define kbhit(x) (x != 0x0F)
#define Power_Down 0b00010000

volatile uint32_t Voltage_ADC_value = 0x00;
volatile uint32_t Current_ADC_value = 0x00;
volatile uint16_t Measure_count = 0x00;
volatile uint8_t ADC_Flag = 0x00;
uint16_t EEMEM u3c_voltage;
uint16_t EEMEM offset_error;
uint16_t EEMEM gain_error;
uint8_t EEMEM  Cal_current;
uint8_t EEMEM  Cal_voltage;
uint16_t EEMEM Set_Voltage;
uint16_t EEMEM Set_Current;

void ADC_init (void)
{

    DDRB &= ~( 1 << PB4 );// Voltage+
    DDRB &= ~( 1 << PB3 );// Voltage-
    DDRB &= ~( 1 << PB2 );// Current

    PORTB &= ~( 1 << PB4 );// Voltage+
    PORTB &= ~( 1 << PB3 );//Voltage-
    PORTB &= ~( 1 << PB2 );// Current

    ADMUX = 0x00;  // cleanup
    ADCSRA = 0x00; // cleanup
    ADCSRB = 0x00; // cleanup
    DIDR0 = 0x00;  // cleanup

    Voltage_Chanel;

    ADCSRA |= ( 1 << ADEN ); // ADC Enable
    ADCSRA |= ( 1 << ADIE ); // ADC Interrupt Enable
    ADCSRA |= ( 1 << ADPS0 ) | ( 1 << ADPS1 ) | ( 1 << ADPS2 ); // ADC Prescaler 128
    ADCSRA |= ( 1 << ADATE ); // ADC Auto Trigger Enable
    DIDR0 = ( 1 << ADC1D ) | ( 1 << ADC2D ) | ( 1 << ADC3D);
    //ADCSRB = ( 1 << BIN );

    MCUCR |= ( 1 << SE ); //sleep enable
    MCUCR |= ( 1 << SM0);// ADC noise reduction
    PRR = ( 1 << PRTIM1 ) | ( 1 << PRTIM0 ) | ( 1 << PRUSI );// power off

    ADC_Start;

}

ISR ( ADC_vect )
{

    if ( Measure_count == 0) // change channel
        Voltage_Chanel;


    if ( Measure_count == 267) // change channel
        Current_Chanel;


    if ((Measure_count > 10) && (Measure_count < 267)) // discard first 0-9 measurements
        Voltage_ADC_value += ( ADC );  // 255 samples


    if ((Measure_count > 277) && (Measure_count < 534)) // discard first 267-277 measurements
        Current_ADC_value += ADC; // 255 samples


    if (Measure_count <= 534) // count to 535 and stop
        Measure_count++;

}


uint8_t keyboard_write_read (uint8_t data)
{
    uint8_t value = 0;

    if (!i2c_start( Key_I2C_Adress + I2C_WRITE)) // start comunication
    {
        i2c_write(data);
        i2c_rep_start( Key_I2C_Adress + I2C_READ ); // start comunication
        value = i2c_readNak(); // contains stop condition
        //i2c_stop();
    }
    else
        return 0;

    return value;
}

uint8_t keyboard_scan (void)
{
    return keyboard_write_read(0b00001111);
}


uint8_t keyboard_row (const uint8_t *ROW)
{
    uint8_t value = 0x00;

    for (uint8_t i = 0; i < 4; i++)
    {
        value = keyboard_write_read(0b00001111 | ( 0b10000000 >> i));

        if ((value & 0x0F) == 0x0F)
        {
            return ROW[i];
        }

    }

    return 0;

}

uint8_t keyboard_button (uint8_t data)
{
    //P0 A B C D
    //P1 3 6 9 #
    //P2 2 5 8 0
    //P3 1 4 7 *

    static const uint8_t P0_ROW [] = {'A','B','C','D'};
    static const uint8_t P1_ROW [] = {'3','6','9','#'};
    static const uint8_t P2_ROW [] = {'2','5','8','0'};
    static const uint8_t P3_ROW [] = {'1','4','7','*'};
    static uint8_t keyboard_trigger;

    if (kbhit(data) && (keyboard_trigger == 0x00) ) // button press, button not pressed before
    {
        keyboard_trigger = 0x01; // set status button pressed

        switch (data) // check which button pressed
        {
        case 0x0E :

            return keyboard_row(P0_ROW);
            break;

        case 0x0D :

            return keyboard_row(P1_ROW);
            break;

        case 0x0B :

            return keyboard_row(P2_ROW);
            break;

        case 0x07 :

            return keyboard_row(P3_ROW);
            break;

        default :
            return 0;

        }

    }

    if ( !kbhit(data) && (keyboard_trigger == 0x01)) // release button
    {
        keyboard_trigger = 0; // cleanup
    }

    return 0;


}


void mcp4725_send_data( uint32_t data, uint8_t I2C_addr )

{

    i2c_start_wait(I2C_addr); // start comunication

    data = data << 4;

    i2c_write(0b01100000);// save data to Eeprom mcp4725
    i2c_write((data & 0xFF00) >> 8);  // Write H data to mcp4725
    i2c_write((data & 0x00F0));  // Write L data to mcp4725

    i2c_stop();


}

void measure_voltage_current (uint32_t *voltage_current,const uint16_t gain_error,const uint16_t u3c_voltage)
{
    Voltage_ADC_value = Voltage_ADC_value >> 4;
    Current_ADC_value = Current_ADC_value >> 4;

    // [0] - voltage [1] - current [2] - u3c_voltage

    voltage_current[0] = ((uint32_t)Voltage_ADC_value * Ref_Volt) >> 14;
    voltage_current[1] = ((uint32_t)Current_ADC_value * Ref_Volt) >> 14;

    voltage_current[0] *= Divide;

    voltage_current[0] *= 1000;
    voltage_current[0] /= gain_error;


    voltage_current[2] = voltage_current[1];

    voltage_current[1] = abs( (uint32_t)u3c_voltage - (voltage_current[1]) ); //
    voltage_current[1] *= 100;
    voltage_current[1] /= Gain;


}


int main(void)
{

    char LCD_RAM[5];
    char Input_Voltage[4];
    char Input_Current[4];
    uint32_t Voltage_Current[] = {0,0,0}; // [0] - voltage [1] - current [2] - u3c_voltage
    uint32_t Set_Voltage_ram = 0; // cleanup
   static uint32_t Set_Current_ram = 0; // cleanup
    volatile uint16_t offset_error_ram = 0x00; // default value
    volatile uint16_t gain_error_ram = 1000;   // default value
    volatile uint16_t u3c_voltage_ram = 4500;  // default value
    uint8_t Keyboard = 0; // cleanup
    uint8_t i = 0;  // cleanup
    //uint16_t j = 0;

    i2c_init();
    ADC_init();
    LCD_init();
    LCD_light_ON = True;
    LCD_write_string("Voltage ",LCD_char_addr(0x00));
    LCD_write_string("Current ",LCD_char_addr(0x09));

    _delay_ms(1000);

    LCD_write(Clear_Display,LCD_command);

    _delay_ms(100);

    LCD_write_string("U:00.0V ",LCD_char_addr(0x00));
    LCD_write_string("I:00.0A ",LCD_char_addr(0x09));

    if (eeprom_read_byte(&Cal_voltage) == 1)
    {
        gain_error_ram = eeprom_read_word(&gain_error);
        Set_Voltage_ram = eeprom_read_word(&Set_Voltage);

        int_to_char((uint16_t)Set_Voltage_ram,LCD_RAM);
        LCD_write_char(LCD_RAM[2],LCD_char_addr(0x02));
        LCD_write(LCD_RAM[3],LCD_data);
        LCD_write('.',LCD_data);
        LCD_write(LCD_RAM[4],LCD_data);

    }


    if (eeprom_read_byte(&Cal_current) == 1)
    {
        offset_error_ram = eeprom_read_word(&offset_error);
        u3c_voltage_ram = eeprom_read_word(&u3c_voltage);
        Set_Current_ram = eeprom_read_word(&Set_Current);

        int_to_char((uint16_t)Set_Current_ram,LCD_RAM);
        LCD_write_char(LCD_RAM[2],LCD_char_addr(0x0B));
        LCD_write(LCD_RAM[3],LCD_data);
        LCD_write('.',LCD_data);
        LCD_write(LCD_RAM[4],LCD_data);


    }




    sei(); // interrupt enable

    // Insert code

    while(1)
    {

        if (Measure_count == 535)
        {
            cli();

            measure_voltage_current(Voltage_Current,gain_error_ram,u3c_voltage_ram);

            if (Voltage_Current[0] <= offset_error_ram + 2)
                Voltage_Current[0] = abs(Voltage_Current[0] - offset_error_ram);

            int_to_char((uint16_t)Voltage_Current[0],LCD_RAM);

            // LCD_write(LCD_char_addr(0x40),LCD_command); // set cursor position
            LCD_write_char(LCD_RAM[0],LCD_char_addr(0x40));
            LCD_write(LCD_RAM[1],LCD_data);
            LCD_write('.',LCD_data);
            LCD_write(LCD_RAM[2],LCD_data);
            LCD_write(LCD_RAM[3],LCD_data);
            LCD_write(LCD_RAM[4],LCD_data);
            LCD_write('V',LCD_data);
            LCD_write(' ',LCD_data);
            LCD_write(' ',LCD_data);


            int_to_char((uint16_t)Voltage_Current[1],LCD_RAM);

            LCD_write(LCD_RAM[0],LCD_data);
            LCD_write(LCD_RAM[1],LCD_data);
            LCD_write('.',LCD_data);
            LCD_write(LCD_RAM[2],LCD_data);
            LCD_write(LCD_RAM[3],LCD_data);
            LCD_write(LCD_RAM[4],LCD_data);
            LCD_write('A',LCD_data);

            Keyboard = keyboard_scan();
            Keyboard = keyboard_button(Keyboard);

            switch (Keyboard)
            {

            case 'A' : // set Voltage

            {
                i = 0; // cleanup
                Input_Voltage[0] = '0'; // cleanup
                Input_Voltage[1] = '0'; // cleanup
                Input_Voltage[2] = '0'; // cleanup
                LCD_write_string("00.0V ",LCD_char_addr(0x02)); // cleanup disp
                LCD_write(LCD_char_addr(0x05),LCD_command); // set cursor position
                LCD_write(Cursor_ON_blinking,LCD_command);  //

                while ( i < 4)
                {

                    Keyboard = keyboard_scan();
                    Keyboard = keyboard_button(Keyboard);

                    if ((Keyboard != 'A') && (Keyboard != 'B') && (Keyboard != 'C') && (Keyboard != 'D') && (Keyboard != '#') && (Keyboard != '*') && (Keyboard != 0))
                    {

                        switch (i)
                        {
                        case 0:
                        {
                            Input_Voltage[2] = Keyboard;
                            LCD_write_char(Input_Voltage[2],LCD_char_addr(0x05));
                            LCD_write(LCD_char_addr(0x05),LCD_command); // set cursor
                            break;
                        }

                        case 1:
                        {
                            Input_Voltage[1] = Input_Voltage[2];
                            Input_Voltage[2] = Keyboard;
                            LCD_write_char(Input_Voltage[1],LCD_char_addr(0x03));
                            LCD_write_char(Input_Voltage[2],LCD_char_addr(0x05));
                            LCD_write(LCD_char_addr(0x05),LCD_command); // set cursor
                            break;
                        }
                        case 2:
                        {
                            Input_Voltage[0] = Input_Voltage[1];
                            Input_Voltage[1] = Input_Voltage[2];
                            Input_Voltage[2] = Keyboard;
                            LCD_write_char(Input_Voltage[0],LCD_char_addr(0x02));
                            LCD_write_char(Input_Voltage[1],LCD_char_addr(0x03));
                            LCD_write_char(Input_Voltage[2],LCD_char_addr(0x05));
                            LCD_write(LCD_char_addr(0x05),LCD_command); // set cursor

                            break;

                        }

                        }

                        i++;

                    }

                    if ((Keyboard == '#'))
                    {

                        Set_Voltage_ram = (uint16_t)atoi(Input_Voltage);
                        if ((Set_Voltage_ram >= min_voltage) && (Set_Voltage_ram <= max_voltage))
                        {
                            eeprom_write_word(&Set_Voltage,Set_Voltage_ram);
                            Set_Voltage_ram *= 1000;
                            Set_Voltage_ram /= 6;
                            Set_Voltage_ram *= 4096;
                            Set_Voltage_ram /= 50000;

                            //zapni vystup na pozadovanu hodnotu
                            mcp4725_send_data(Set_Voltage_ram,MCP4725_Voltage_I2C_Adress);
                            break;
                        }
                        else
                        {
                            Set_Voltage_ram = 0;
                            i = 4;
                        }
                    }

                    if ((i == 4))
                    {
                        LCD_write_string("00.0V ",LCD_char_addr(0x02)); // cleanup disp
                        LCD_write(LCD_char_addr(0x05),LCD_command); // set cursor position
                        Input_Voltage[0] = '0'; // cleanup
                        Input_Voltage[1] = '0'; // cleanup
                        Input_Voltage[2] = '0'; // cleanup
                        i = 0; // cleanup

                    }

                }
                LCD_write(Cursor_Off,LCD_command);

                break;

            }


            case 'B' : // set current

            {
                i = 0; // cleanup
                Input_Current[0] = '0'; // cleanup
                Input_Current[1] = '0'; // cleanup
                Input_Current[2] = '0'; // cleanup
                LCD_write_string("00.0A ",LCD_char_addr(0x0B)); // cleanup disp
                LCD_write(LCD_char_addr(0x0E),LCD_command); // set cursor position
                LCD_write(Cursor_ON_blinking,LCD_command);  //

                while ( i < 4)
                {

                    Keyboard = keyboard_scan();
                    Keyboard = keyboard_button(Keyboard);

                    if ((Keyboard != 'A') && (Keyboard != 'B') && (Keyboard != 'C') && (Keyboard != 'D') && (Keyboard != '#') && (Keyboard != '*') && (Keyboard != 0))
                    {

                        switch (i)
                        {
                        case 0:
                        {
                            Input_Current[2] = Keyboard;
                            LCD_write_char(Input_Current[2],LCD_char_addr(0x0E));
                            LCD_write(LCD_char_addr(0x0E),LCD_command); // set cursor
                            break;
                        }

                        case 1:
                        {
                            Input_Current[1] = Input_Current[2];
                            Input_Current[2] = Keyboard;
                            LCD_write_char(Input_Current[1],LCD_char_addr(0x0C));
                            LCD_write_char(Input_Current[2],LCD_char_addr(0x0E));
                            LCD_write(LCD_char_addr(0x0E),LCD_command); // set cursor
                            break;
                        }
                        case 2:
                        {
                            Input_Current[0] = Input_Current[1];
                            Input_Current[1] = Input_Current[2];
                            Input_Current[2] = Keyboard;
                            LCD_write_char(Input_Current[0],LCD_char_addr(0x0B));
                            LCD_write_char(Input_Current[1],LCD_char_addr(0x0C));
                            LCD_write_char(Input_Current[2],LCD_char_addr(0x0E));
                            LCD_write(LCD_char_addr(0x0E),LCD_command); // set cursor

                            break;

                        }

                        }

                        i++;

                    }

                    if ((Keyboard == '#'))
                    {

                        Set_Current_ram = (uint16_t)atoi(Input_Current);
                        if ((Set_Current_ram >= min_current) && (Set_Current_ram <= max_current))

                        {
                            //uint32_t pomocna = 0;
                            eeprom_write_word(&Set_Current,Set_Current_ram);
                            //pomocna = Set_Current_ram*20;
                            Set_Current_ram *= 100;
                            //Set_Current_ram += (Set_Current_ram/10);
                            //Set_Current_ram *= 9;
                            //Set_Current_ram /= 10;
                            Set_Current_ram = abs (u3c_voltage_ram - Set_Current_ram);
                           // Set_Current_ram -= pomocna;
                            Set_Current_ram *= 4096;
                            Set_Current_ram /= 500;
                            if ((uint8_t)(Set_Current_ram%10) > 5)
                            {
                                Set_Current_ram /= 10;
                                Set_Current_ram += 1;
                            }
                            else
                               Set_Current_ram /= 10;

                            //zapni vystup na pozadovanu hodnotu

                            mcp4725_send_data((uint16_t)Set_Current_ram,MCP4725_Current_I2C_Adress);
                            break;
                        }
                        else
                        {
                            Set_Current_ram = 0;
                            i = 4;
                        }
                    }

                    if ((i == 4))
                    {
                        LCD_write_string("00.0A ",LCD_char_addr(0x0B)); // cleanup disp
                        LCD_write(LCD_char_addr(0x0E),LCD_command); // set cursor position
                        Input_Current[0] = '0'; // cleanup
                        Input_Current[1] = '0'; // cleanup
                        Input_Current[2] = '0'; // cleanup
                        i = 0;

                    }



                }
                LCD_write(Cursor_Off,LCD_command);

                break;

            }

            case 'C' : // current callibration

            {

                mcp4725_send_data(4095,MCP4725_Current_I2C_Adress); // set 0A output  current

                _delay_ms(100);

                cli();

                Measure_count = 0; // Cleanup
                Voltage_ADC_value = 0; // cleanup
                Current_ADC_value = 0; // cleanup

                sei();

                while (Measure_count != 535)
                {

                    asm volatile ("sleep");

                }

                measure_voltage_current(Voltage_Current,1000,0);

                eeprom_write_word(&offset_error,Voltage_Current[0]);
                eeprom_write_word(&u3c_voltage,Voltage_Current[2]);

                u3c_voltage_ram = Voltage_Current[2];
                offset_error_ram = Voltage_Current[0];

                eeprom_write_byte(&Cal_current,1);

                int_to_char((uint16_t)Voltage_Current[2],LCD_RAM);

                LCD_write_char(LCD_RAM[0],LCD_char_addr(0x49)); // show call value
                LCD_write(LCD_RAM[1],LCD_data);
                LCD_write('.',LCD_data);
                LCD_write(LCD_RAM[2],LCD_data);
                LCD_write(LCD_RAM[3],LCD_data);
                LCD_write(LCD_RAM[4],LCD_data);
                LCD_write('A',LCD_data);

                cli();

                _delay_ms(5000);

                break;


            }

            case 'D' : // voltage callibration

            {
                i = 0;// cleanup
                Input_Voltage[0] = '0'; // cleanup
                Input_Voltage[1] = '0'; // cleanup
                Input_Voltage[2] = '0'; // cleanup

                LCD_write_string("00.0V ",LCD_char_addr(0x02)); // cleanup disp
                LCD_write(LCD_char_addr(0x05),LCD_command); // set cursor position
                LCD_write(Cursor_ON_blinking,LCD_command);
                mcp4725_send_data(4095,MCP4725_Voltage_I2C_Adress); // set max output voltage

                while ( i < 4)
                {

                    Keyboard = keyboard_scan();
                    Keyboard = keyboard_button(Keyboard);

                    if ((Keyboard != 'A') && (Keyboard != 'B') && (Keyboard != 'C') && (Keyboard != 'D') && (Keyboard != '#') && (Keyboard != '*') && (Keyboard != 0))
                    {

                        switch (i)
                        {
                        case 0:
                        {
                            Input_Voltage[2] = Keyboard;
                            LCD_write_char(Input_Voltage[2],LCD_char_addr(0x05));
                            LCD_write(LCD_char_addr(0x05),LCD_command); // set cursor
                            break;
                        }

                        case 1:
                        {
                            Input_Voltage[1] = Input_Voltage[2];
                            Input_Voltage[2] = Keyboard;
                            LCD_write_char(Input_Voltage[1],LCD_char_addr(0x03));
                            LCD_write_char(Input_Voltage[2],LCD_char_addr(0x05));
                            LCD_write(LCD_char_addr(0x05),LCD_command); // set cursor
                            break;
                        }
                        case 2:
                        {
                            Input_Voltage[0] = Input_Voltage[1];
                            Input_Voltage[1] = Input_Voltage[2];
                            Input_Voltage[2] = Keyboard;
                            LCD_write_char(Input_Voltage[0],LCD_char_addr(0x02));
                            LCD_write_char(Input_Voltage[1],LCD_char_addr(0x03));
                            LCD_write_char(Input_Voltage[2],LCD_char_addr(0x05));
                            LCD_write(LCD_char_addr(0x05),LCD_command); // set cursor

                            break;

                        }

                        }

                        i++;

                    }

                    if ((Keyboard == '#'))
                    {

                        Set_Voltage_ram = (uint16_t)atoi(Input_Voltage);
                        if ((Set_Voltage_ram >= 290) && (Set_Voltage_ram <= 320 ))

                        {
                            cli();

                            Measure_count = 0; // Cleanup
                            Voltage_ADC_value = 0; // cleanup
                            Current_ADC_value = 0; // cleanup

                            sei();

                            while (Measure_count != 535)
                            {

                                asm volatile ("sleep");

                            }

                            measure_voltage_current(Voltage_Current,1000,0);

                            int_to_char((uint16_t)Voltage_Current[0],LCD_RAM);

                            LCD_write_char(LCD_RAM[0],LCD_char_addr(0x40));
                            LCD_write(LCD_RAM[1],LCD_data);
                            LCD_write('.',LCD_data);
                            LCD_write(LCD_RAM[2],LCD_data);
                            LCD_write(LCD_RAM[3],LCD_data);
                            LCD_write(LCD_RAM[4],LCD_data);
                            LCD_write('V',LCD_data);

                            gain_error_ram = (Voltage_Current[0]*10) / Set_Voltage_ram;
                            eeprom_write_word(&gain_error,gain_error_ram);

                            eeprom_write_byte(&Cal_voltage,1);

                            int_to_char((uint16_t)gain_error_ram,LCD_RAM);

                            LCD_write_char(LCD_RAM[0],LCD_char_addr(0x49));//show call value
                            LCD_write(LCD_RAM[1],LCD_data);
                            LCD_write('.',LCD_data);
                            LCD_write(LCD_RAM[2],LCD_data);
                            LCD_write(LCD_RAM[3],LCD_data);
                            LCD_write(LCD_RAM[4],LCD_data);
                            LCD_write(' ',LCD_data);

                            cli();

                            _delay_ms(5000);

                            break;
                        }
                        else
                        {
                            Set_Voltage_ram = 0;
                            i = 4;
                        }
                    }

                    if ((i == 4))
                    {
                        LCD_write_string("00.0V ",LCD_char_addr(0x02)); // cleanup disp
                        LCD_write(LCD_char_addr(0x05),LCD_command); // set cursor position
                        Input_Voltage[0] = '0'; // cleanup
                        Input_Voltage[1] = '0'; // cleanup
                        Input_Voltage[2] = '0'; // cleanup
                        i = 0;

                    }



                }
                LCD_write(Cursor_Off,LCD_command);

                break;

            }




            }

            Voltage_ADC_value = 0; // cleanup
            Current_ADC_value = 0; // cleanup
            Measure_count = 0;     // cleanup

            sei();

        }


        asm volatile ("sleep");


    }


    return 0;
}
